#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QTime>
#include <QDebug>

// send an UDP packet to an host
void MainWindow::sendUdpPacket(QByteArray const &packet, QHostAddress addr)
{
	QUdpSocket *sock = new QUdpSocket;
	sock->writeDatagram(packet, addr, UDP_PORT);
	delete(sock);
}

// wait for and get an UDP packet from host
QByteArray MainWindow::receiveUdpPacket(quint16 timeout, QHostAddress &sender, quint16 &port)
{
	QByteArray res;
	res.clear();

	// wait for incoming answers
	QTime timer;
	timer.start();
	while(timer.elapsed() < timeout)
	{
		// if there are pending packets....
		if(udpSocket->hasPendingDatagrams())
		{
			// build res array

			// read it
			res.resize(udpSocket->pendingDatagramSize());
			udpSocket->readDatagram(res.data(), res.size(), &sender, &port);

			return res;
		}
	}
	return res;
}

QByteArray MainWindow::receiveUdpPacket(quint16 timeout)
{
	QHostAddress host;
	quint16 port;
	return receiveUdpPacket(timeout, host, port);
}


void MainWindow::discoverDevices()
{
	// build UDP discovery packet
	QByteArray arr(1, UDP_FIND);
	sendUdpPacket(arr, QHostAddress::Broadcast);

	// clear droplist
	ui->deviceCombo->clear();

	// wait for incoming answers
	QByteArray res;
	do
	{
		// try to read the packet
		QHostAddress host;
		quint16 port;
		res = receiveUdpPacket(500, host, port);
		qDebug() << "Packet size:" << res.size() << "\n";

		// if not done, leave
		if(!res.size())
			break;

		// if packet has a single byte, it's our own query packet
		// just discard it
		if(res.size() <= 1)
			continue;

		QString deb;
		for(int k = 0; k < res.size(); k++)
			deb += QString("0x%1").arg((int)res[k], 0, 16) + ":";
		qDebug() << deb;

		quint8 const *resPtr = (quint8 const *)res.data();
		if(*resPtr++ == UDP_FIND)
		{
			quint8 nDevices = *resPtr++;
			for(quint8 iDev = 0; iDev < nDevices; iDev++)
			{
				// device number
				quint8 devNum = *resPtr++;

				// device name
				QString name = (const char *)resPtr;
				resPtr += strlen((const char *)resPtr) + 1;
				QList<QVariant> list;
				list << devNum << host.toIPv4Address();
				ui->deviceCombo->addItem(name, list);
			}
		}

	}
	while(true);
}

// get device values
void MainWindow::getDeviceValues(int)
{
	// get current device index
	int comboIdx = ui->deviceCombo->currentIndex();
	if(comboIdx == -1)
		return;

	// get device IP and number from combobox data
	QList<QVariant> list = ui->deviceCombo->currentData().value<QList<QVariant> >();
	qint8 device = list[0].value<qint8>();
	QHostAddress addr(list[1].value<quint32>());

	// build the needed query UDP packet
	QByteArray packet;
	packet.append(UDP_GETLIGHT).append(device);

	// send the packet
	sendUdpPacket(packet, addr);

	// read response back
	packet = receiveUdpPacket(200);

	qDebug() << "Packet size:" << packet.size() << "\n";

	// response packet must be 6 bytes long
	if(packet.size() == 6 && (quint8)packet[0] == UDP_GETLIGHT)
	{
		quint8 const *packP = (quint8 const *)packet.data() + 2;
		ui->redSlider->setValue(*packP++);
		ui->greenSlider->setValue(*packP++);
		ui->blueSlider->setValue(*packP++);
		ui->whiteSlider->setValue(*packP++);
	}

}

// set device values
void MainWindow::setDeviceValues(int)
{
	qint8 r, g, b, w;

	// get current device index
	int comboIdx = ui->deviceCombo->currentIndex();
	if(comboIdx == -1)
		return;

	// get slider values
	r = ui->redSlider->value();
	g = ui->greenSlider->value();
	b = ui->blueSlider->value();
	w = ui->whiteSlider->value();

	// get device IP and number from combobox data
	QList<QVariant> list = ui->deviceCombo->currentData().value<QList<QVariant> >();
	qint8 device = list[0].value<qint8>();
	QHostAddress addr(list[1].value<quint32>());

	// build the needed query UDP packet
	QByteArray packet;
	packet.append(UDP_SETLIGHT).append(device);
	packet.append(r).append(g).append(b).append(w);

	// send the packet
	sendUdpPacket(packet, addr);

}

// set device name
void MainWindow::setDeviceName(QString newName)
{

}


MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent), ui(new Ui::MainWindow)
{
	ui->setupUi(this);
	ui->redSlider->setMinimum(0);
	ui->redSlider->setMaximum(255);
	ui->greenSlider->setMinimum(0);
	ui->greenSlider->setMaximum(255);
	ui->blueSlider->setMinimum(0);
	ui->blueSlider->setMaximum(255);
	ui->whiteSlider->setMinimum(0);
	ui->whiteSlider->setMaximum(255);

	// initialize UDP socket
	udpSocket = new QUdpSocket(this);
	udpSocket->bind(UDP_PORT, QUdpSocket::ShareAddress);

	// discover available devices
	discoverDevices();

	connect(ui->redSlider, SIGNAL(sliderMoved(int)), this, SLOT(setDeviceValues(int)));
	connect(ui->greenSlider, SIGNAL(sliderMoved(int)), this, SLOT(setDeviceValues(int)));
	connect(ui->blueSlider, SIGNAL(sliderMoved(int)), this, SLOT(setDeviceValues(int)));
	connect(ui->whiteSlider, SIGNAL(sliderMoved(int)), this, SLOT(setDeviceValues(int)));

	connect(ui->redSlider, SIGNAL(sliderReleased()), this, SLOT(setDeviceValues()));
	connect(ui->greenSlider, SIGNAL(sliderReleased()), this, SLOT(setDeviceValues()));
	connect(ui->blueSlider, SIGNAL(sliderReleased()), this, SLOT(setDeviceValues()));
	connect(ui->whiteSlider, SIGNAL(sliderReleased()), this, SLOT(setDeviceValues()));

	connect(ui->deviceCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(getDeviceValues(int)));
}

MainWindow::~MainWindow()
{
	delete ui;
}
