/*
  Octopus.h - Library to handle Fishino Octopus I/O expander boards
  Copyright (c) 2016 Massimo Del Fedele. All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  VERSION 1.0.0 - INITIAL VERSION
  VERSION 5.0.0 - Converted to new library format
*/
#ifndef __OCTOPUS_H
#define __OCTOPUS_H

#include <Arduino.h>

#define Octopus __octopus()

// struct to cache some MCP registers
typedef struct
{
	uint16_t GPIO;
	uint16_t IODIR;
	uint16_t GPPU;
	uint16_t IOINT;
} MCPCache;

class OctopusClass
{
	private:
	
		// connected boards and their addresses
		uint8_t nBoards;
		uint8_t *addresses;
		
		// cached MCP registers
		MCPCache *mcpCaches;
		
		// low-level MCP register access
		void writeMCPRegister8(uint8_t mcpAddr, uint8_t regAddr, uint8_t val);
		void writeMCPRegister16(uint8_t mcpAddr, uint8_t regAddr, uint16_t val);
		uint16_t readMCPRegister16(uint8_t mcpAddr, uint8_t regAddr);

		// low-level PCA register access
		void writePCARegister8(uint8_t pcaAddr, uint8_t regAddr, uint8_t val);
		void writePCARegister16(uint8_t pcaAddr, uint8_t regAddr, uint16_t val);
		uint8_t readPCARegister8(uint8_t pcaAddr, uint8_t regAddr);
		uint16_t readPCARegister16(uint8_t pcaAddr, uint8_t regAddr);

		// check if the PWM part is at given address
		// return true on success, false otherwise
		bool detectPWM(uint8_t addr);
		
		// sets the 2 PWM values for an output
		// values can be 0..4095
		// special values of 0xffff are reserved to turn fully ON or OFF
		// the output (OFF has precedence over ON)
		void setPWMValues(uint8_t port, uint16_t on, uint16_t off);
		
		// check if the digital part is at given address
		// return true on success, false otherwise
		bool detectIO(uint8_t addr);
	
		// initialize and count the connected boards
		uint8_t initialize(void);

	protected:

	public:

		// constructor
		OctopusClass();

		// destructor
		~OctopusClass();
		
		// return number of boards found
		uint8_t getNumBoards(void) const { return nBoards; }
		
		// return number of available I/O
		uint8_t getNumIO(void) const { return nBoards * 16; }
		
		// set pwm frequency for a single connected board
		// valid values 24 Hz...1526 Hz
		void setPWMFreq(uint8_t board, uint16_t freq);
		
		// set pwm frequency for ALL connected boards
		void setPWMFreq(uint16_t freq);
		
		// pwm output
		void analogWrite(uint8_t port, uint16_t val, bool invert = false);
		
		// digital I/O
		void pinMode(uint8_t port, uint8_t mode);
		bool digitalRead(uint8_t port);
		void digitalWrite(uint8_t port, bool value);
		
		// read/write all digital pins of given board at once
		uint16_t digitalReadAll(uint8_t board);
		void digitalWriteAll(uint8_t board, uint16_t val);

};

OctopusClass &__octopus();

#endif
