#include <Arduino.h>

#include "FishinoXPT2046.h"

static SPISettings spiSettings;

// S A2 A1 A0 MODE SER/DFR PD1 PD0
#define XPT2046_S				0b10000000

#define XPT2046_SER_TEMP0		0b00000000
#define XPT2046_SER_Y			0b00010000
#define XPT2046_SER_VBAT		0b00100000
#define XPT2046_SER_Z1			0b00110000
#define XPT2046_SER_Z2			0b01000000
#define XPT2046_SER_X			0b01010000
#define XPT2046_SER_AUX			0b01100000
#define XPT2046_SER_X_TEMP		0b01110000

#define XPT2046_DFR_Y			0b00010000
#define XPT2046_DFR_Z1			0b00110000
#define XPT2046_DFR_Z2			0b01000000
#define XPT2046_DFR_X			0b01010000

#define XPT2046_SER				0b00000100
#define XPT2046_DFR				0b00000000

#define XPT2046_PWR_OFF			0b00000000
#define XPT2046_PWR_ADC			0b00000001
#define XPT2046_PWR_REF			0b00000010
#define XPT2046_PWR_ON			0b00000011

#define XPT2046_NUMVALS			10
#define XPT2046_DISCARD			2

int16_t _vals[XPT2046_NUMVALS];

// touch interrupt service routine
// (probably we can do without it..)
void FishinoXPT2046_ISR(void)
{
//	Serial << "ISR\n";
	FishinoXPT2046._touched = true;
}

// constructor
FishinoXPT2046Class::FishinoXPT2046Class()
{
	_connected = false;
	_cs = 0;
	_irq = 0;
	
	_width = 0;
	_height = 0;

	_rotation = 0;
	
	_x = _y = 0;
	_z = 0;
	_pressure = 0;
	_touched = false;
	
	// setup default calibration values
	calibrate();

	// setup SPI settings
	// (can't be done as a static value on 32 bit boards)
	spiSettings = SPISettings(2000000, MSBFIRST, SPI_MODE0);
	
	// connect to default pins
	// use connect() function to change them
	connect(TOUCH_CS, TOUCH_IRQ);
}

// destructor
FishinoXPT2046Class::~FishinoXPT2046Class()
{
	// detach interrupt routine
	detachInterrupt(digitalPinToInterrupt(_irq));
}

// read some points, discards the extremes
// and take the mean of internal ones
int16_t FishinoXPT2046Class::readVal(uint8_t what)
{
	// prepare to read first value
	SPI.transfer(XPT2046_S | what | XPT2046_PWR_ADC);
	for(int i = 0; i < XPT2046_NUMVALS; i++)
		_vals[i] = SPI.transfer16(XPT2046_S | what | XPT2046_PWR_ADC) >> 3;
	
	// re-power off
	SPI.transfer16(XPT2046_S);
	
	// sort read values
	for(int i = 0; i < XPT2046_NUMVALS - 1; i++)
		for(int j = i + 1; j < XPT2046_NUMVALS; j++)
			if(_vals[i] < _vals[j])
			{
				int16_t tmp = _vals[i];
				_vals[i] = _vals[j];
				_vals[j] = tmp;
			}
			
	// take the mean of inner values
	int16_t retVal = 0;
	for(int i = XPT2046_DISCARD ; i < XPT2046_NUMVALS - XPT2046_DISCARD; i++)
		retVal += _vals[i];
	return retVal / (XPT2046_NUMVALS - 2 * XPT2046_DISCARD);
}
		
// update values from touch
bool FishinoXPT2046Class::update(void)
{
	if(!_connected || !_touched)
		return false;
	
	// detach interrupt routine while processing
	detachInterrupt(digitalPinToInterrupt(_irq));
	
	// start a transaction
	SPI.beginTransaction(spiSettings);
	digitalWrite(_cs, LOW);
	
	// read all needed values, taking means and discarding bads
	int32_t z1 = readVal(XPT2046_DFR_Z1);
	int32_t z2 = readVal(XPT2046_DFR_Z2);
	int32_t z = z1 + 4095 - z2;
	if(z < 0)
		z = 0;
	
	if(z < XPT2046_Z_THRESHOLD)
	{
		_x = -1;
		_y = -1;
		_z = 0;
		_touched = false;
	}
	else
	{
		_x = readVal(XPT2046_DFR_X);
		_y = readVal(XPT2046_DFR_Y);
		z = (((4096L * z1) / _x) * 4096L) / (z2 - z1);
		if(z < 0)
			z = 0;
		_z = (uint16_t)z;
	}

	// end the transaction
	digitalWrite(_cs, HIGH);
	SPI.endTransaction();
	
	// re-attach interrupt at processing end
	attachInterrupt(digitalPinToInterrupt(_irq), FishinoXPT2046_ISR, FALLING);
	if(_z < XPT2046_Z_THRESHOLD)
		_touched = false;

	return _touched;
}

// disconnect from hardware ports
void FishinoXPT2046Class::disconnect(void)
{
	if(!_connected)
		return;

	// detach interrupt routine
	detachInterrupt(digitalPinToInterrupt(_irq));
	
	// set cs pin to input
	pinMode(_cs, INPUT);
}
		
// connect to hardware ports
void FishinoXPT2046Class::connect(uint8_t cs, uint8_t irq)
{
	// if already connected, disconnect first
	disconnect();
		
	// store connection pins
	_cs = cs;
	_irq = irq;
	
	_x = _y = 0;
	_z = 0;
	_pressure = 0;
	_touched = false;
	
	pinMode(_cs, OUTPUT);
	pinMode(_irq, INPUT_PULLUP);
	attachInterrupt(digitalPinToInterrupt(_irq), FishinoXPT2046_ISR, FALLING);
	
	SPI.begin();
	
	// initialize the controller
	SPI.beginTransaction(spiSettings);
	digitalWrite(_cs, LOW);
	
	// power down
	SPI.transfer(XPT2046_S);
	SPI.transfer(0);

	digitalWrite(_cs, HIGH);
	SPI.endTransaction();

	_connected = true;
}

// set calibration values
void FishinoXPT2046Class::calibrate(uint16_t w, uint16_t h, int16_t ax, int16_t bx, int16_t ay, int16_t by)
{
	_width = w;
	_height = h;
	_ax = ax;
	_bx = bx;
	_ay = ay;
	_by = by;
}

// set default calibration values
// (see comments on include file)
void FishinoXPT2046Class::calibrate()
{
	_width = XPT2046_WIDTH;
	_height = XPT2046_HEIGHT;
	_ax = XPT2046_AX;
	_bx = XPT2046_BX;
	_ay = XPT2046_AY;
	_by = XPT2046_BY;
}

// set display rotation (counterclockwise)
void FishinoXPT2046Class::setRotation(uint8_t rot)
{
	// rot must be between 0 and 3
	_rotation = rot & 0x03;
}
		
		
// check if being touched
bool FishinoXPT2046Class::touching(void)
{
	return update();
}

// read raw coordinates (as returned by XPT2046 chip)
void FishinoXPT2046Class::readRaw(uint16_t &x, uint16_t &y, int16_t &z)
{
	x = _x;
	y = _y;
	z = _z;
}

// read calibrated coordinates (and pressure value)
void FishinoXPT2046Class::read(uint16_t &x, uint16_t &y, uint16_t &pressure)
{
	// convert raw coordinates to screen coordinates
	// on unrotated screen
	int32_t xx = (((int32_t)_x + _bx) << 3) / _ax;
	if(xx < 0)
		xx = 0;
	else if(xx >= XPT2046_WIDTH)
		xx = XPT2046_WIDTH - 1;
	int32_t yy = (((int32_t)_y + _by) << 3) / _ay;
	if(yy < 0)
		yy = 0;
	else if(yy >= XPT2046_HEIGHT)
		yy = XPT2046_HEIGHT - 1;
	
	// apply rotation
	switch(_rotation)
	{
		case 0:
		default:
			x = (uint16_t)xx;
			y = (uint16_t)yy;
			break;
		case 1:
			x = (uint16_t)yy;
			y = (uint16_t)(XPT2046_WIDTH - xx - 1);
			break;
		case 2:
			x = (uint16_t)(XPT2046_WIDTH - xx - 1);
			y = (uint16_t)(XPT2046_HEIGHT - yy - 1);
			break;
		case 3:
			x = (uint16_t)(XPT2046_HEIGHT - yy - 1);
			y = (uint16_t)xx;
			break;
	}
	
	pressure = _z;
}

void FishinoXPT2046Class::read(uint16_t &x, uint16_t &y)
{
	uint16_t z;
	read(x, y, z);
}

TSPoint FishinoXPT2046Class::read(void)
{
	TSPoint p;
	read(p.x, p.y, p.pressure);
	return p;
}


FishinoXPT2046Class &__GetFishinoXPT2046(void)
{
	static FishinoXPT2046Class x;
	return x;
}
