//
// Released to the public domain! Enjoy!
// v02 - 26/04/2017
// v03 - 18/06/2017
// for Fishino32 by Edoardo Mecchina
//
/////////////////////////
//
// The RTC is set according to the Timezone
//
// The DST (with Italian Rules) is calculated and added to
//     getRtc(date_time_t *dt)
//     getAlarm(date_time_t *dt)
// responses
//
// Use	RTC32_OUT_NONE
//			RTC32_OUT_ALARM
//			RTC32_OUT_CLOCK
// in setupRtc(...) to enable/disable Clock/Alarm output on pin D10
//
/////////////////////////

#include <Arduino.h>
#include "Fishino32RTC.h"

////////////////////////////////////////////////////////////////////////////////
// RTC Conversion routines

static uint8_t bcd2bin(uint8_t val)
{
	return val - 6 * (val >> 4);
}
static uint8_t bin2bcd(uint8_t val)
{
	return val + 6 * (val / 10);
}
//
// Epoch Routines
//
#define EPOCH_DOW 6
#define SECS_PER_DAY 86400
#define UNIX_TO_EPOCH_SECONDS 946684800

static unsigned short days[4][12] =
{
	{   0,  31,  60,  91, 121, 152, 182, 213, 244, 274, 305, 335},
	{ 366, 397, 425, 456, 486, 517, 547, 578, 609, 639, 670, 700},
	{ 731, 762, 790, 821, 851, 882, 912, 943, 974, 1004, 1035, 1065},
	{1096, 1127, 1155, 1186, 1216, 1247, 1277, 1308, 1339, 1369, 1400, 1430},
};

unsigned int day_of_week(const int year, const int month, const int day, const int hour, const int minute, const int second)
{
	uint32_t day_now;

	day_now = year / 4 * (365 * 4 + 1) + days[year % 4][month - 1] + day - 1;
	return (EPOCH_DOW + day_now) % 7;						// Jan 1, 2000 is a Saturday, i.e. returns 6
}
//////////////////////////////////
// year  		0 - 99
// month		1 - 12
// day			1 - 31
// hour			0 - 23
// minute		0 - 59
// second		0 - 59
// wday			0 -  6		Weekday: 0 = Domenica, 1 = Luned, ...
// dst			0 -  1 		Daylight Saving (Ora Legale):  0 = No - 1 = S
///////////////////////////////////
unsigned int date_time_set(date_time_t *dt, const int year, const int month, const int day, const int hour, const int minute, const int second, const int wday, const int dst)
{
	uint32_t day_now;
	uint32_t epoch;

	if (year >= 2000)
		dt->year = year - 2000;
	else
		dt->year = year;    // 0-99
	//
	day_now = dt->year / 4 * (365 * 4 + 1) + days[year % 4][month - 1] + day - 1;
	//
	dt->second = second;  // 0-59
	dt->minute = minute;  // 0-59
	dt->hour = hour;    	// 0-23
	dt->day = day; 				// 1-31
	dt->month = month; 		// 1-12
	//
	if (wday >= 0)
		dt->wday = wday;
	else
		dt->wday = (EPOCH_DOW + day_now) % 7;		// Jan 1, 2000 is a Saturday, i.e. returns 6
	//
	epoch = ((day_now * 24 + hour) * 60 + minute) * 60 + second;
	epoch -= 3600 * dst;

	epoch_to_date_time(dt, epoch);
	return epoch;
}

unsigned int date_time_to_epoch(date_time_t *dt)
{
	uint32_t epoch;

	//
	unsigned int second = dt->second;  		// 0-59
	unsigned int minute = dt->minute;  		// 0-59
	unsigned int hour   = dt->hour;    		// 0-23
	unsigned int day    = dt->day - 1; 		// 0-30
	unsigned int month  = dt->month - 1; 	// 0-11
	unsigned int year   = dt->year;    		// 0-99
	//
	epoch = (((year / 4 * (365 * 4 + 1) + days[year % 4][month] + day) * 24 + hour) * 60 + minute) * 60 + second;
	//
	return epoch;
}

void epoch_to_date_time(date_time_t *dt, unsigned int epoch)
{
	unsigned int years, year, month;

	dt->wday = (EPOCH_DOW + epoch / SECS_PER_DAY) % 7;		// Jan 1, 2000 is a Saturday, i.e. returns 6

	dt->second = epoch % 60;
	epoch /= 60;
	dt->minute = epoch % 60;
	epoch /= 60;
	dt->hour   = epoch % 24;
	epoch /= 24;

	years = epoch / (365 * 4 + 1) * 4;
	epoch %= 365 * 4 + 1;
	for (year = 3; year > 0; year--)
	{
		if (epoch >= days[year][0])
			break;
	}

	for (month = 11; month > 0; month--)
	{
		if (epoch >= days[year][month])
			break;
	}

	dt->year  = years + year;
	dt->month = month + 1;
	dt->day   = epoch - days[year][month] + 1;
}

unsigned int date_time_to_unix(date_time_t *dt)
{
	unsigned int second = dt->second;  	// 0-59
	unsigned int minute = dt->minute;  	// 0-59
	unsigned int hour   = dt->hour;    	// 0-23
	unsigned int day    = dt->day - 1; 	// 0-30
	unsigned int month  = dt->month - 1; // 0-11
	unsigned int year   = dt->year;    	// 0-99
	return UNIX_TO_EPOCH_SECONDS + (((year / 4 * (365 * 4 + 1) + days[year % 4][month] + day) * 24 + hour) * 60 + minute) * 60 + second;
}

int date_time_is_dst(date_time_t *dt)
{
	int previousSunday = dt->day - (dt->wday);
	int result;

	switch (dt->month)
	{
		case 1:
		case 2:
		case 11:
		case 12:
			result = 0;
			break;
		case 3:
			if (previousSunday > 25)
			{
				if (dt->wday == 0)
					result = dt->hour >= 2;
				else
					result = 1;
			}
			else
				result = 0;
			break;
		case 10:
			if (previousSunday < 25)
				result = 1;
			else
			{
				if (dt->wday == 0)
					result = dt->hour < 2;
				else
					result = 0;
			}
			break;
		default:
			result = 1;
			break;
	}
	return result;
}

////////////////////////////////////////////////////////////////////////////////
// RtcPic32Class implementation
////////////////////////////////////////////////////////////////////////////////
// constructor
RtcPic32Class::RtcPic32Class()
{
	// Constructor code here
	_vec = _RTCC_VECTOR;			// #define _RTCC_VECTOR		 25
	_irq = _RTCC_IRQ;					// #define _RTCC_IRQ			 30
	_ipl = _RTCC_IPL_IPC;			// #define _RTCC_IPL_IPC		2		(2)
	_spl = _RTCC_SPL_IPC;			// #define _RTCC_SPL_IPC		0   (1)

	_isrHandler = NULL;
	_rtcInterruptHandler = NULL;
	_interruptsEnabled = false;
}
//
// destructor
RtcPic32Class::~RtcPic32Class()
{
	// just in case...
	disableInterrupts();
}

void RtcPic32Class::setupRtc(uint32_t mode)
{
	struct rtctime_s time;
	struct rtcdate_s date;

	time.val   = 0;
	time.sec   = 0;
	time.min   = 0;
	time.hour  = 0;
	date.val   = 0;
	date.wday  = 6;
	date.day   = 1;
	date.month = 1;
	date.year  = 0;

	disableInterrupts();
	SYSKEY = 0x0;
	SYSKEY = 0xaa996655; 													// 0xBF80F230 - write first unlock key to SYSKEY
	SYSKEY = 0x556699aa; 													// 0xBF80F230 - write second unlock key to SYSKEY
//	RTCCONSET = RTCCON_RTCWREN;
	SETOFFSET(_rtccon)->rtcwen = 1;								// set RTCWREN in RTCCONSET
	while ((_rtccon->rtcsync) != 0)
		;								// wait for RTCSYNC
	_rtctime->val = time.val;
	_rtcdate->val = date.val;
	_alrmtime->val = time.val;
	_alrmdate->val = date.val;
	CLROFFSET(_rtccon)->val = RTC32_OUT_MASK;			// Disable Clock Output // 0 = Alarm, 0 = NO Clock on Output
	SETOFFSET(_rtccon)->val = mode;								// Enable Clock Output // 0 = Alarm, 1 = Clock on Output
	SETOFFSET(_rtccon)->on = 1;										// turn ON the RTCC
	while ((_rtccon->on) == 0)
		;										// wait RTCC to go ON
	CLROFFSET(_rtccon)->rtcwen = 1;								// clear RTCWREN in RTCCONSET
	enableInterrupts();
}

uint32_t RtcPic32Class::isRtcRunning(void)
{
	uint32_t retval;

	retval = _rtccon->on;
	return retval;
}

void RtcPic32Class::setRtc(date_time_t *dt)
{
	struct rtctime_s time;
	struct rtcdate_s date;

	time.sec   = bin2bcd(dt->second);
	time.min   = bin2bcd(dt->minute);
	time.hour  = bin2bcd(dt->hour);
	date.wday  = dt->wday;												// Jan 1, 2000 is a Saturday, i.e. returns 6
	date.day   = bin2bcd(dt->day);
	date.month = bin2bcd(dt->month);
	date.year  = bin2bcd(dt->year);

// starting critical sequence										// MUST be three consecutive instructions!!
	disableInterrupts();
	SYSKEY = 0x0;
	SYSKEY = 0xaa996655; 													// 0xBF80F230 - write first unlock key to SYSKEY
	SYSKEY = 0x556699aa; 													// 0xBF80F230 - write second unlock key to SYSKEY
//	RTCCONSET = RTCCON_RTCWREN;
	SETOFFSET(_rtccon)->rtcwen = 1;								// set RTCWREN in RTCCONSET
	enableInterrupts();
// end critical sequence

//	noInterrupts();
	while ((_rtccon->rtcsync) != 0)
		;							// wait for RTCSYNC
	_rtctime->val = time.val;
	_rtcdate->val = date.val;
	CLROFFSET(_rtccon)->rtcwen = 1;								// clear RTCWREN in RTCCONSET
//	interrupts();
}

uint32_t RtcPic32Class::getRtc(date_time_t *dt)
{
	unsigned int epoch, epochNow;
	struct rtctime_s time;
	struct rtcdate_s date;

	while ((_rtccon->rtcsync) != 0)
		;		// wait for not ALRMSYNC
	time.val = _rtctime->val;
	date.val = _rtcdate->val;

	dt->second = bcd2bin(time.sec);
	dt->minute = bcd2bin(time.min);
	dt->hour = bcd2bin(time.hour);
	dt->wday =  bcd2bin(date.wday);
	dt->day = bcd2bin(date.day);
	dt->month = bcd2bin(date.month);
	dt->year = bcd2bin(date.year);				// RTC DateTime

	epoch = date_time_to_epoch(dt);				// RTC Epoch
	epochNow = epoch + (3600 * date_time_is_dst(dt));		// Now Epoch
	epoch_to_date_time(dt, epochNow);										// NOW DateTime
	return epoch;
}

uint32_t RtcPic32Class::isAlarmRunning(void)
{
	uint32_t retval;

	retval = _rtcalrm->alrmen;
	return retval;
}

void RtcPic32Class::setAlarm(date_time_t *dt, uint32_t mode, bool enable)
{
	struct alrmtime_s time;
	struct alrmdate_s date;

	time.sec   = bin2bcd(dt->second);
	time.min   = bin2bcd(dt->minute);
	time.hour  = bin2bcd(dt->hour);
	date.wday  = bin2bcd(dt->wday);
	date.day   = bin2bcd(dt->day);
	date.month = bin2bcd(dt->month);

	while ((_rtcalrm->alrmsync) != 0)
		;										// Wait for ALRM SYNC
	while ((_rtccon->rtcsync) != 0)
		;											// Wait for RTC SYNC
	CLROFFSET(_rtcalrm)->val = 0x0000EFFF;  						// Clear Alarms
	_alrmtime->val = time.val;
	_alrmdate->val = date.val;
	//
//		SETOFFSET(_rtcalrm)->val = 0x0003;								// Alarm Repeat
	if (!enable)
		SETOFFSET(_rtcalrm)->val = (mode);								// Set and diasble Alarm
	else
		SETOFFSET(_rtcalrm)->val = (mode | 0x00008000);		// Set and enable Alarm
}

void RtcPic32Class::resetAlarm()
{
	while ((_rtcalrm->alrmsync) != 0)
		;										// Wait for ALRMSYNC
	CLROFFSET(_rtcalrm)->val = 0x0000EFFF;  						// Clear Alarms
}

uint32_t RtcPic32Class::getAlarm(date_time_t *dt)
{
	unsigned int epoch, epochNow;
	struct alrmtime_s time;
	struct alrmdate_s date;

	while ((_rtcalrm->alrmsync) != 0)
		;							// Wait for not ALRMSYNC
	time.val = _alrmtime->val;
	date.val = _alrmdate->val;

	dt->second = bcd2bin(time.sec);
	dt->minute = bcd2bin(time.min);
	dt->hour = bcd2bin(time.hour);
	dt->wday =  bcd2bin(date.wday);
	dt->day = bcd2bin(date.day);
	dt->month = bcd2bin(date.month);
	dt->year = bcd2bin(date.year);				// RTC DateTime

	epoch = date_time_to_epoch(dt);				// RTC Epoch
	epochNow = epoch + (3600 * date_time_is_dst(dt));		// Now Epoch
	epoch_to_date_time(dt, epochNow);										// NOW DateTime
	return epoch;
}

////////////////////////////////////////////////////////////////////////////////
// RtcPic32Class Interrupt Routines
////////////////////////////////////////////////////////////////////////////////
static RtcPic32Class *_rtc_instance = NULL;
void __attribute__((interrupt(), nomips16)) _rtcInterruptHandler(void)
{
	_rtc_instance->isr();
}

// internal interrupt handler
void RtcPic32Class::isr(void)
{
	// if we've set an handler, just call it
	if (_rtcInterruptHandler)
		_rtcInterruptHandler(_handlerParameter);

	// clear interrupt flag on exit
	clearIntFlag(_irq);
}

////////////////////////////////////////////////////////////////////////////////
// attach an interrupt handler to Rtc
////////////////////////////////////////////////////////////////////////////////
void RtcPic32Class::_attachInterrupt(RTC32_INTERRUPT_HANDLER isr, void *param)
{
	// if interrupts enabled, disable them
	if (_interruptsEnabled)
		disInterrupts();

	// store user's interrupt handler
	_handlerParameter = param;
	_rtcInterruptHandler = isr;

	// enable interrupts again
	enInterrupts();
}

// detach the interrupt handler from RTC
void RtcPic32Class::detachInterrupt()
{
	// if interrupts enabled, disable them
	if (_interruptsEnabled)
		disInterrupts();

	// remove user's interrupt handler
	_rtcInterruptHandler = NULL;
}

// enable interrupts
void RtcPic32Class::enInterrupts(void)
{
	// if we don't have any handler or already enabled, just do nothing
	if (!_rtcInterruptHandler || _interruptsEnabled)
		return;

	// connect external ISR handler
	setIntVector(_vec, _isrHandler);

	// setup interrupt priority
	setIntPriority(_vec, _ipl, _spl);

	// clear any pending interrupt
	clearIntFlag(_irq);

	// enable interrupts
	setIntEnable(_irq);

	// mark interrupts as enabled
	_interruptsEnabled = true;
}

// disable interrupts
void RtcPic32Class::disInterrupts(void)
{
	// if not enabled, just do nothing
	if (!_interruptsEnabled)
		return;

	// disable interrupts and vector
	clearIntEnable(_irq);
	setIntPriority(_vec, 0, 0);
	clearIntVector(_vec);

	// mark interrupts as disabled
	_interruptsEnabled = false;
}

////////////////////////////////////////////////////////////////////////////////
struct _RTC32_Initializer
{
	int _vec;
	int _irq;
	int _ipl;
	int _spl;
	volatile void *_rtccon;
	volatile void *_rtcalrm;
	volatile void *_rtctime;
	volatile void *_rtcdate;
	volatile void *_alrmtime;
	volatile void *_alrmdate;
	void (*isrHandler)(void);
	RtcPic32Class **_instance;
};

const _RTC32_Initializer _RTC32_Initializers[] =
{
	{ _RTCC_VECTOR, _RTCC_IRQ, _RTCC_IPL_IPC, _RTCC_SPL_IPC, &RTCCON, &RTCALRM, &RTCTIME, &RTCDATE, &ALRMTIME, &ALRMDATE, _rtcInterruptHandler, &_rtc_instance },
};

void _RTC32_Initialize(RtcPic32Class &t, uint8_t index)
{
	_RTC32_Initializer const &i = _RTC32_Initializers[index];

	t._vec = i._vec;
	t._irq = i._irq;
	t._ipl = i._ipl;
	t._spl = i._spl;
	t._rtccon = (rtccon_s *)i._rtccon;
	t._rtcalrm = (rtcalrm_s *)i._rtcalrm;
	t._rtctime = (rtctime_s *)i._rtctime;
	t._rtcdate = (rtcdate_s *)i._rtcdate;
	t._alrmtime = (alrmtime_s *)i._alrmtime;
	t._alrmdate = (alrmdate_s *)i._alrmdate;
	t._isrHandler = i.isrHandler;
	*i._instance = &t;
}

////////////////////////////////////////////////////////////////////////////////
RtcPic32Class &_rtc1()
{
	static RtcPic32Class t;
	static bool initialized = false;

	if (!initialized)
	{
		_RTC32_Initialize(t, 0);
		initialized = true;
	}
	return t;
}
